function [ sa_tables, sa_output, b ] = sa_main_mod_2021_NicotineModel( num_samps )
    %{ 
        This function is the main function of the PHM. It uses experts' inputs
        to produce samples using a Latin hypercube method needed to run the
        Monte Carlo simulation. This function also produces model outputs.
        The experts' inputs are located in the rawvals matrix, and it is organized as follow: 
        Columns indicate the minimum, 5th, 25th, 50th, 75th and 95th percentiles, and maximum values. Rows are organized as follow: 
        Row 1-7: female: Fraction of smokers that quit in first year of policy. 
        Row 8-15: female: Fraction of smokers that quit after first year of policy.
        Row 16-21: male: Fraction of smokers that quit in first year of policy. 
        Row 22-28: male: Fraction of smokers that quit after first year of policy. 
        Row 29-35: female: Fraction of quitters who switch to non-combusted in first year of policy. 
        Row 36-42: female: Fraction of quitters who switch to non-combusted after first year of policy. 
        Row 43-49: male: Fraction of quitters who switch to non-combusted in first year of policy. 
        Row 50-56: male: Fraction of quitters who switch to non-combusted after first year of policy. 
        Row 57-63: female: Fraction of continuing smokers who become dual users in first year of policy. 
        Row 65-70: female: Fraction of continuing smokers who become dual users after first year of policy. 
        Row 71-77: male: Fraction of continuing smokers who become dual users in first year of policy. 
        Row 78-84: male: Fraction of continuing smokers who become dual users after first year of policy. 
        Row 85-91: female: smoking initiation reduction factor in first year of policy. 
        Row 92-98: female: smoking initiation reduction factor after first year of policy. 
        Row 99-105: male: smoking initiation reduction factor in first year of policy. 
        Row 106-112: male: smoking initiation reduction factor after first year of policy. 
        Row 113-119: female: fraction of dissuaded initiates who become noncombustible users in first year of policy. 
        Row 120-126: female: fraction of dissuaded initiates who become noncombustible users after first year of policy. 
        Row 127-133: female: fraction of dissuaded initiates who become noncombustible users in first year of policy. 
        Row 134-140: female: fraction of dissuaded initiates who become noncombustible users after first year of policy
        
    %}
    
    % Note: Illicit trade scaling factors used to adjust cessation rates from 
    % experts' opinion are defined in lines 197 to 198.
    
    %===============================================================%================================
    policy_func=@policy_sa;  % calling the policy function 
       
	% CHANGE THIS TO CONTROL HOW SAMPLING IS HANDLED:
    % @do_samps will sample each expert in order. @lhs_lin_interp_func will
    % treat the expert as an additional parameter and perform a random sample
    
    samps_func=@do_samps;
    
    
    pctiles=[0 5 25 50 75 95 100]/100;
    
    rawvals = [ ...
    0.200	0.250	0.350	0.500	0.650	0.750	0.800
    0.040	0.060	0.080	0.100	0.150	0.200	0.250
    0.035	0.040	0.050	0.065	0.080	0.095	0.100
    0.200	0.250	0.350	0.475	0.550	0.650	0.700
    0.200	0.300	0.450	0.600	0.700	0.800	0.900
    0.035	0.070	0.135	0.285	0.335	0.535	0.635
    0.070	0.200	0.300	0.500	0.600	0.800	0.900
    0.200	0.250	0.350	0.500	0.650	0.750	0.800
    0.050	0.070	0.100	0.150	0.200	0.250	0.300
    0.035	0.040	0.045	0.050	0.065	0.075	0.080
    0.200	0.250	0.350	0.475	0.550	0.650	0.700
    0.200	0.300	0.400	0.500	0.600	0.700	0.800
    0.035	0.070	0.135	0.185	0.235	0.285	0.335
    0.070	0.200	0.300	0.500	0.600	0.800	0.900
    0.200	0.250	0.350	0.500	0.650	0.750	0.800
    0.040	0.060	0.080	0.100	0.150	0.200	0.250
    0.035	0.040	0.050	0.065	0.080	0.095	0.100
    0.200	0.250	0.350	0.475	0.550	0.650	0.700
    0.200	0.300	0.450	0.600	0.700	0.800	0.900
    0.035	0.070	0.135	0.285	0.335	0.535	0.635
    0.070	0.200	0.300	0.500	0.600	0.800	0.900
    0.200	0.250	0.350	0.500	0.650	0.750	0.800
    0.050	0.070	0.100	0.150	0.200	0.250	0.300
    0.035	0.040	0.045	0.050	0.065	0.075	0.080
    0.200	0.250	0.350	0.475	0.550	0.650	0.700
    0.200	0.300	0.400	0.500	0.600	0.700	0.800
    0.035	0.070	0.135	0.185	0.235	0.285	0.335
    0.070	0.200	0.300	0.500	0.600	0.800	0.900
    0.200	0.250	0.400	0.500	0.600	0.750	0.800
    0.200	0.250	0.500	0.600	0.700	0.850	0.900
    0.300	0.450	0.500	0.550	0.600	0.650	0.700
    0.400	0.450	0.550	0.650	0.750	0.850	0.900
    0.300	0.500	0.600	0.700	0.750	0.800	0.900
    0.070	0.150	0.400	0.650	0.750	0.850	0.950
    0.050	0.100	0.200	0.250	0.350	0.500	0.700
    0.200	0.250	0.400	0.500	0.600	0.750	0.800
    0.200	0.250	0.500	0.600	0.700	0.850	0.900
    0.500	0.550	0.600	0.700	0.730	0.760	0.800
    0.400	0.450	0.550	0.650	0.750	0.850	0.900
    0.300	0.500	0.600	0.700	0.750	0.800	0.900
    0.070	0.150	0.400	0.650	0.750	0.850	0.950
    0.050	0.100	0.200	0.250	0.350	0.500	0.700
    0.200	0.250	0.400	0.500	0.600	0.750	0.800
    0.200	0.250	0.500	0.600	0.700	0.850	0.900
    0.300	0.450	0.500	0.550	0.600	0.650	0.700
    0.400	0.450	0.550	0.650	0.750	0.850	0.900
    0.300	0.500	0.600	0.700	0.750	0.800	0.900
    0.070	0.150	0.400	0.650	0.750	0.850	0.950
    0.050	0.100	0.200	0.250	0.350	0.500	0.700
    0.200	0.250	0.400	0.500	0.600	0.750	0.800
    0.200	0.250	0.500	0.600	0.700	0.850	0.900
    0.500	0.550	0.600	0.700	0.730	0.760	0.800
    0.400	0.450	0.550	0.650	0.750	0.850	0.900
    0.300	0.500	0.600	0.700	0.750	0.800	0.900
    0.070	0.150	0.400	0.650	0.750	0.850	0.950
    0.050	0.100	0.200	0.250	0.350	0.500	0.700
    0.100	0.150	0.300	0.450	0.600	0.750	0.800
    0.100	0.200	0.300	0.600	0.800	0.850	0.950
    0.400	0.450	0.500	0.600	0.750	0.800	0.850
    0.500	0.550	0.650	0.750	0.850	0.900	0.950
    0.400	0.500	0.650	0.800	0.850	0.900	0.950
    0.120	0.200	0.300	0.600	0.700	0.800	0.920
    0.100	0.250	0.400	0.500	0.600	0.750	0.900
    0.100	0.150	0.300	0.450	0.600	0.750	0.800
    0.100	0.200	0.300	0.600	0.800	0.850	0.950
    0.020	0.050	0.100	0.150	0.175	0.200	0.300
    0.500	0.550	0.650	0.750	0.850	0.900	0.950
    0.400	0.500	0.650	0.800	0.850	0.900	0.950
    0.120	0.150	0.200	0.300	0.500	0.600	0.920
    0.100	0.250	0.400	0.500	0.600	0.750	0.900
    0.100	0.150	0.300	0.450	0.600	0.750	0.800
    0.100	0.200	0.300	0.600	0.800	0.850	0.950
    0.400	0.450	0.500	0.600	0.750	0.800	0.850
    0.500	0.550	0.650	0.750	0.850	0.900	0.950
    0.400	0.500	0.650	0.800	0.850	0.900	0.950
    0.120	0.200	0.300	0.600	0.700	0.800	0.920
    0.100	0.250	0.400	0.500	0.600	0.750	0.900
    0.100	0.150	0.300	0.450	0.600	0.750	0.800
    0.100	0.200	0.300	0.600	0.800	0.850	0.950
    0.020	0.050	0.100	0.150	0.175	0.200	0.300
    0.500	0.550	0.650	0.750	0.850	0.900	0.950
    0.400	0.500	0.650	0.800	0.850	0.900	0.950
    0.120	0.150	0.200	0.300	0.500	0.600	0.920
    0.100	0.250	0.400	0.500	0.600	0.750	0.900
    0.100	0.150	0.200	0.300	0.400	0.450	0.500
    0.500	0.600	0.700	0.750	0.850	0.900	0.950
    0.400	0.420	0.450	0.500	0.600	0.670	0.700
    0.050	0.100	0.200	0.275	0.350	0.450	0.500
    0.050	0.100	0.150	0.200	0.300	0.400	0.500
    0.050	0.100	0.150	0.300	0.600	0.650	0.700
    0.050	0.100	0.200	0.250	0.300	0.400	0.500
    0.100	0.150	0.200	0.300	0.400	0.450	0.500
    0.500	0.600	0.700	0.750	0.850	0.900	0.950
    0.300	0.350	0.400	0.450	0.560	0.620	0.650
    0.050	0.100	0.200	0.275	0.350	0.450	0.500
    0.050	0.100	0.150	0.200	0.300	0.400	0.500
    0.050	0.100	0.150	0.300	0.600	0.650	0.700
    0.020	0.080	0.150	0.200	0.250	0.350	0.450
    0.100	0.150	0.200	0.300	0.400	0.450	0.500
    0.500	0.600	0.700	0.750	0.850	0.900	0.950
    0.400	0.420	0.450	0.500	0.600	0.670	0.700
    0.050	0.100	0.200	0.275	0.350	0.450	0.500
    0.050	0.100	0.150	0.200	0.300	0.400	0.500
    0.050	0.100	0.150	0.300	0.600	0.650	0.700
    0.050	0.100	0.200	0.250	0.300	0.400	0.500
    0.100	0.150	0.200	0.300	0.400	0.450	0.500
    0.500	0.600	0.700	0.750	0.850	0.900	0.950
    0.300	0.350	0.400	0.450	0.560	0.620	0.650
    0.050	0.100	0.200	0.275	0.350	0.450	0.500
    0.050	0.100	0.150	0.200	0.300	0.400	0.500
    0.050	0.100	0.150	0.300	0.600	0.650	0.700
    0.020	0.080	0.150	0.200	0.250	0.350	0.450
    0.050	0.100	0.150	0.200	0.250	0.400	0.450
    0.250	0.350	0.450	0.650	0.750	0.800	0.900
    0.700	0.730	0.750	0.800	0.850	0.900	1.000
    0.050	0.100	0.200	0.300	0.400	0.500	0.550
    0.300	0.500	0.650	0.800	0.850	0.900	0.950
    0.100	0.250	0.400	0.650	0.700	0.800	0.900
    0.000	0.030	0.070	0.100	0.200	0.250	0.300
    0.050	0.100	0.150	0.200	0.250	0.400	0.450
    0.250	0.350	0.450	0.650	0.750	0.800	0.900
    0.600	0.650	0.680	0.700	0.800	0.900	1.000
    0.050	0.100	0.200	0.300	0.400	0.500	0.550
    0.300	0.500	0.650	0.800	0.850	0.900	0.950
    0.100	0.250	0.400	0.650	0.700	0.800	0.900
    0.050	0.100	0.150	0.200	0.300	0.400	0.500
    0.050	0.100	0.150	0.200	0.250	0.400	0.450
    0.250	0.350	0.450	0.650	0.750	0.800	0.900
    0.700	0.730	0.750	0.800	0.850	0.900	1.000
    0.050	0.100	0.200	0.300	0.400	0.500	0.550
    0.300	0.500	0.650	0.800	0.850	0.900	0.950
    0.100	0.250	0.400	0.650	0.700	0.800	0.900
    0.000	0.030	0.070	0.100	0.200	0.250	0.300
    0.050	0.100	0.150	0.200	0.250	0.400	0.450
    0.250	0.350	0.450	0.650	0.750	0.800	0.900
    0.600	0.650	0.680	0.700	0.800	0.900	1.000
    0.050	0.100	0.200	0.300	0.400	0.500	0.550
    0.300	0.500	0.650	0.800	0.850	0.900	0.950
    0.100	0.250	0.400	0.650	0.700	0.800	0.900
    0.050	0.100	0.150	0.200	0.300	0.400	0.500     ];
    
    
    
    for i=1:7
        row_id = i:7:140;
        vals(:,:,i) = rawvals(row_id,:);
    end
    
    % Additional parameter to implement the Illicit Trade scenarios (3.8%, 5.9% and 21%), 0% means no illicit trade (general model)
    IT_val = 0; %0 0.038 0.059  0.21 0.30
    for i=1:7
        vals(1:2,:,i)=(1-IT_val)*vals(1:2,:,i)/0.035; % Females
        vals(3:4,:,i)=(1-IT_val)*vals(3:4,:,i)/0.035; % Males
        %vals(1:2,:,i)=(1-IT_val)*vals(1:2,:,i)/0.045872078; %0.045071675; %0.045048303; % Females
        %vals(3:4,:,i)=(1-IT_val)*vals(3:4,:,i)/0.049609657; %0.044718556; %0.044694735; % Males
    end
    
    
    
    rng(77); %rng(1);
    
    [ samps, prob_vals, val_sets ] = samps_func(num_samps, vals, pctiles);
    
    
    input{1}='policy_count';
    input{2}=0;
    input{3}='clear';
    input{4}=true;
    input{5}='raw_inputs';
    input{6}='InputData_Nicotine_Baseline2021.xlsx';
    input{7}='load_rsrc_func';
    input{8}=@two_p_ind_prev_mat.load_raw_data_initbyyear;
    input{9}='pad_calc';
    input{10}=true;
    input{11}='num_years';
    input{12}=80;  % from 2021 to 2100: total 80 years
    input{13}='policy_func';
    input{14}=@wrapper_func;
    input{15}='save_output';
    input{16}=false;
    input{17}='mortality_func';
    input{18}=@mort_func;
    input{19}='transit_func';
    input{20}=@trans_probs_initbyyear;
    %input{20}=@trans_probs_initbyyear_mentholbanSA;
    
    for jjj=1:10
        sa_output.adult_prev_c1{jjj}=zeros(input{12}, num_samps);
        sa_output.adult_prev_c2{jjj}=zeros(input{12}, num_samps);
        sa_output.adult_prev_cc{jjj}=zeros(input{12}, num_samps);
        sa_output.adult_prev_ctot{jjj}=zeros(input{12}, num_samps);
        
        sa_output.prev1824_c1{jjj}=zeros(input{12}, num_samps);
        sa_output.prev1824_c2{jjj}=zeros(input{12}, num_samps);
        sa_output.prev1824_cc{jjj}=zeros(input{12}, num_samps);
        sa_output.prev1824_ctot{jjj}=zeros(input{12}, num_samps);
        
        sa_output.prev2544_c1{jjj}=zeros(input{12}, num_samps);
        sa_output.prev2544_c2{jjj}=zeros(input{12}, num_samps);
        sa_output.prev2544_cc{jjj}=zeros(input{12}, num_samps);
        sa_output.prev2544_ctot{jjj}=zeros(input{12}, num_samps);
        
        sa_output.prev4564_c1{jjj}=zeros(input{12}, num_samps);
        sa_output.prev4564_c2{jjj}=zeros(input{12}, num_samps);
        sa_output.prev4564_cc{jjj}=zeros(input{12}, num_samps);
        sa_output.prev4564_ctot{jjj}=zeros(input{12}, num_samps);
        
        sa_output.prev65100_c1{jjj}=zeros(input{12}, num_samps);
        sa_output.prev65100_c2{jjj}=zeros(input{12}, num_samps);
        sa_output.prev65100_cc{jjj}=zeros(input{12}, num_samps);
        sa_output.prev65100_ctot{jjj}=zeros(input{12}, num_samps);
        
        sa_output.new_cess{jjj}=zeros(input{12}, num_samps);
        sa_output.new_init{jjj}=zeros(input{12}, num_samps);
        sa_output.pad{jjj}=zeros(input{12}, num_samps);
        sa_output.padP1{jjj}=zeros(input{12}, num_samps);
        sa_output.padP2{jjj}=zeros(input{12}, num_samps);
        sa_output.padDUAL{jjj}=zeros(input{12}, num_samps);
        sa_output.lifeyears{jjj}=zeros(input{12}, num_samps);
        sa_output.ad_morb_qaly_lost{jjj}=zeros(input{12}, num_samps);
        
        % For validation:
        sa_output.pop_tot{jjj}=zeros(input{12}, num_samps);
        sa_output.death_f{jjj}=zeros(input{12}, num_samps);
        sa_output.death_m{jjj}=zeros(input{12}, num_samps);
        sa_output.births{jjj}=zeros(input{12}, num_samps);
        sa_output.pop_female1534{jjj}=zeros(input{12}, num_samps);
        sa_output.pop1824{jjj}=zeros(input{12}, num_samps);
        sa_output.pop2544{jjj}=zeros(input{12}, num_samps);
        sa_output.pop4564{jjj}=zeros(input{12}, num_samps);
        sa_output.pop65100{jjj}=zeros(input{12}, num_samps);
        
    end
    % -------------- to save some transitions ----------------
    
    for jjj=1:2 % first and subsequent years
        % Cessation
        sa_output.cessprob_f_12{jjj} = zeros(101, num_samps); % cn:fn (f) cessation for 1st product between exclusive smokers
        sa_output.cessprob_m_12{jjj} = zeros(101, num_samps);
        
        sa_output.cessprob_f_42{jjj} = zeros(101, num_samps); % cc:fc (n) cessation for 1st product, still using 2nd product
        sa_output.cessprob_m_42{jjj} = zeros(101, num_samps);
        
        sa_output.cessprob_f_44{jjj} = zeros(101, num_samps); % cc:cf (m) cessation for the 2nd product for dual users
        sa_output.cessprob_m_44{jjj} = zeros(101, num_samps);
        
        sa_output.cessprob_f_45{jjj} = zeros(101, num_samps); % cc:ff (p) cessation for 1st and 2nd product
        sa_output.cessprob_m_45{jjj} = zeros(101, num_samps);
        
    sa_output.cessprob_f_72{jjj} = zeros(101, num_samps); % cf:ff (p) cessation for 1st product, still former for 2nd product
    sa_output.cessprob_m_72{jjj} = zeros(101, num_samps);
 
    % Switching
    sa_output.switchprob_f_15{jjj} = zeros(101, num_samps); % cn:fc (f) switching from 1st product to 2nd product
    sa_output.switchprob_m_15{jjj} = zeros(101, num_samps);	
    sa_output.switchprob_f_69{jjj} = zeros(101, num_samps); % 
    sa_output.switchprob_m_69{jjj} = zeros(101, num_samps);
       
       % Initiation
        sa_output.initprob_f_2{jjj} = zeros(101, num_samps); % nn:cn (b) initiation for the 1st product, never users of 2nd product
        sa_output.initprob_m_2{jjj} = zeros(101, num_samps);
        
        sa_output.initprob_f_4{jjj} = zeros(101, num_samps); % nn:nc (a) initiation for the 2nd product, never users of the 1st product
        sa_output.initprob_m_4{jjj} = zeros(101, num_samps);
        
        sa_output.initprob_f_5{jjj} = zeros(101, num_samps); % nn:cc (e) initiation for dual use
        sa_output.initprob_m_5{jjj} = zeros(101, num_samps);
        
     
    end
    % ------------------------------------------------------------------

    baseline = athena_main_mod_initbyyear('two_p_ind_prev_mat',input{:});
    
    zzz=1;
    b.adult_prev_c1(:,1)=baseline.adult_prev_c1{zzz}(:,1:input{12});
    b.adult_prev_c2(:,1)=baseline.adult_prev_c2{zzz}(:,1:input{12});
    b.adult_prev_cc(:,1)=baseline.adult_prev_cc{zzz}(:,1:input{12});
    b.adult_prev_ctot=b.adult_prev_c1+b.adult_prev_c2-b.adult_prev_cc;
    
    %--------new outputs----------------
    b_aux.adult_prev_n1(:,1)=baseline.adult_prev_n1{zzz}(:,1:input{12});
    b_aux.adult_prev_n2(:,1)=baseline.adult_prev_n2{zzz}(:,1:input{12});
    b_aux.adult_prev_f1(:,1)=baseline.adult_prev_f1{zzz}(:,1:input{12});
    b_aux.adult_prev_f2(:,1)=baseline.adult_prev_f2{zzz}(:,1:input{12});
    
    b_aux.adult_pop_c1(:,1)=baseline.adult_pop_c1{zzz}(:,1:input{12});
    b_aux.adult_pop_c2(:,1)=baseline.adult_pop_c2{zzz}(:,1:input{12});
    b_aux.adult_pop_cc(:,1)=baseline.adult_pop_cc{zzz}(:,1:input{12});
    b_aux.adult_pop_ctot=b_aux.adult_pop_c1+b_aux.adult_pop_c2-b_aux.adult_pop_cc;
    
    b_aux.adult_pop_n1(:,1)=baseline.adult_pop_n1{zzz}(:,1:input{12});
    b_aux.adult_pop_n2(:,1)=baseline.adult_pop_n2{zzz}(:,1:input{12});
    b_aux.adult_pop_f1(:,1)=baseline.adult_pop_f1{zzz}(:,1:input{12});
    b_aux.adult_pop_f2(:,1)=baseline.adult_pop_f2{zzz}(:,1:input{12});
    
    % current users of product 1 regardless of use of product 2
    b_aux.smokers(:,:) = baseline.smokers{zzz}(1:input{12},:);
    %----------------------------------
    for ag=1:5
        b.prev_agegroup_c1_f(:,ag)=baseline.prev_cn_f{zzz,ag}(:,1:input{12})+baseline.prev_cc_f{zzz,ag}(:,1:input{12})+baseline.prev_cf_f{zzz,ag}(:,1:input{12});
        b.prev_agegroup_c2_f(:,ag)=baseline.prev_nc_f{zzz,ag}(:,1:input{12})+baseline.prev_cc_f{zzz,ag}(:,1:input{12})+baseline.prev_fc_f{zzz,ag}(:,1:input{12});
        b.prev_agegroup_cc_f(:,ag)=baseline.prev_cc_f{zzz,ag}(:,1:input{12});
        b.prev_agegroup_ctot_f(:,ag)=b.prev_agegroup_c1_f(:,ag)+b.prev_agegroup_c2_f(:,ag)-b.prev_agegroup_cc_f(:,ag);
        
        b.prev_agegroup_c1_m(:,ag)=baseline.prev_cn_m{zzz,ag}(:,1:input{12})+baseline.prev_cc_m{zzz,ag}(:,1:input{12})+baseline.prev_cf_m{zzz,ag}(:,1:input{12});
        b.prev_agegroup_c2_m(:,ag)=baseline.prev_nc_m{zzz,ag}(:,1:input{12})+baseline.prev_cc_m{zzz,ag}(:,1:input{12})+baseline.prev_fc_m{zzz,ag}(:,1:input{12});
        b.prev_agegroup_cc_m(:,ag)=baseline.prev_cc_m{zzz,ag}(:,1:input{12});
        b.prev_agegroup_ctot_m(:,ag)=b.prev_agegroup_c1_m(:,ag)+b.prev_agegroup_c2_m(:,ag)-b.prev_agegroup_cc_m(:,ag);
        
        b.prev_agegroup_c1(:,ag)=baseline.prev_cn{zzz,ag}(:,1:input{12})+baseline.prev_cc{zzz,ag}(:,1:input{12})+baseline.prev_cf{zzz,ag}(:,1:input{12});
        b.prev_agegroup_c2(:,ag)=baseline.prev_nc{zzz,ag}(:,1:input{12})+baseline.prev_cc{zzz,ag}(:,1:input{12})+baseline.prev_fc{zzz,ag}(:,1:input{12});
        b.prev_agegroup_cc(:,ag)=baseline.prev_cc{zzz,ag}(:,1:input{12});
        b.prev_agegroup_ctot(:,ag)=b.prev_agegroup_c1(:,ag)+b.prev_agegroup_c2(:,ag)-b.prev_agegroup_cc(:,ag);
    end
    
    
    b.new_cessp1(:,1)=[sum(baseline.new_cess_p1_f{zzz}(:,1:input{12})+baseline.new_cess_p1_m{zzz}(:,1:input{12}))]'; % additional results for new quitters
    b.new_cessp2(:,1)=[sum(baseline.new_cess_p2_f{zzz}(:,1:input{12})+baseline.new_cess_p2_m{zzz}(:,1:input{12}))]'; % additional results for new quitters
    b.new_initp1(:,1)=[sum(baseline.new_init_p1_f{zzz}(:,1:input{12})+baseline.new_init_p1_m{zzz}(:,1:input{12}))]'; % additional results for new smoking initiates
    b.new_initp2(:,1)=[sum(baseline.new_init_p2_f{zzz}(:,1:input{12})+baseline.new_init_p2_m{zzz}(:,1:input{12}))]'; % additional results for non combusted initiates
    
    b.new_cessp1E(:,1)=[sum(baseline.new_cess_p1E_f{zzz}(:,1:input{12})+baseline.new_cess_p1E_m{zzz}(:,1:input{12}))]'; % NEW: additional results for new quitters	
    b.new_cessp1Swtch(:,1)=[sum(baseline.new_cess_p1Swtch_f{zzz}(:,1:input{12})+baseline.new_cess_p1Swtch_m{zzz}(:,1:input{12}))]'; % NEW: additional results for new quitters	
    
    %-----new outputs------------------
    b_aux.ave_adult18to100_cess_p1(:,:) = baseline.ave_ad_cess_p1{zzz}(:,:);
    b_aux.ave_adult18to100_cess_p2(:,:) = baseline.ave_ad_cess_p2{zzz}(:,:);
    %b_aux.ave_adult18to100_init_p1(:,:) = baseline.ave_ad_init_p1{zzz}(:,:);
    %b_aux.ave_adult18to100_init_p2(:,:) = baseline.ave_ad_init_p2{zzz}(:,:);
    %b_aux.ave_youth12to17_cess_p1(:,:) = baseline.ave_y_cess_p1{zzz}(:,:);
    %b_aux.ave_youth12to17_cess_p2(:,:) = baseline.ave_y_cess_p2{zzz}(:,:);
    b_aux.ave_youth12to17_init_p1(:,:) = baseline.ave_y_init_p1{zzz}(:,:);
    b_aux.ave_youth12to17_init_p2(:,:) = baseline.ave_y_init_p2{zzz}(:,:);
    %-----------------------------------
    b.pad(:,1)=[sum(baseline.pad_vec_f{zzz}(:,1:input{12})+baseline.pad_vec_m{zzz}(:,1:input{12}))]';
    b.padP1(:,1)=[sum(baseline.padP1_vec_f{zzz}(:,1:input{12})+baseline.padP1_vec_m{zzz}(:,1:input{12}))]'; % additional results for attributable deaths for Product 1
    b.padP2(:,1)=[sum(baseline.padP2_vec_f{zzz}(:,1:input{12})+baseline.padP2_vec_m{zzz}(:,1:input{12}))]'; % additional results for attributable deaths for Product 2
    b.padP1E(:,1)=[sum(baseline.padP1E_vec_f{zzz}(:,1:input{12})+baseline.padP1E_vec_m{zzz}(:,1:input{12}))]'; % additional results for attributable deaths for Exc. Product 1
    b.padP2E(:,1)=[sum(baseline.padP2E_vec_f{zzz}(:,1:input{12})+baseline.padP2E_vec_m{zzz}(:,1:input{12}))]'; % additional results for attributable deaths for Exc. Product 2
    b.padDUAL(:,1)=[sum(baseline.padDUAL_vec_f{zzz}(:,1:input{12})+baseline.padDUAL_vec_m{zzz}(:,1:input{12}))]'; % additional results for DUAL
    
    b.lifeyears(:,1)=baseline.lifeyears{zzz}(2:input{12}+1,:);
    b.ad_morb_qaly_lost(:,1)=baseline.ad_morb_qaly_lost{zzz}(2:input{12}+1,:);
    %----- new outputs------
    b_aux.adult_qaly(:,1)=baseline.adult_qaly{zzz}(2:input{12}+1,:);
    b_aux.age_deaths35to84(:,1)=baseline.age_deaths{zzz}(1:input{12},:);
    %-----------------------
    
    % For validation
    b.pop_tot(:,1)=baseline.pop_tot{zzz}(:,1:input{12});
    b.death_f(:,1)=sum(baseline.death_vec_f{zzz}(:,1:input{12}),1);
    b.death_m(:,1)=sum(baseline.death_vec_m{zzz}(:,1:input{12}),1);
    b.births(:,1)=baseline.births{zzz}(:,1:input{12});
    b.pop_female1534(:,1)=baseline.pop_female1534{zzz,1}(:,1:input{12});
    
    for ag=1:5
        b.pop_agegroup_f(:,ag)=baseline.sub_pop_f{zzz,ag}(:,1:input{12});
        b.pop_agegroup_m(:,ag)=baseline.sub_pop_m{zzz,ag}(:,1:input{12});
        b.pop_agegroup(:,ag)=baseline.sub_pop{zzz,ag}(:,1:input{12});
    end
    
    
    b.transprob_female = squeeze(baseline.trans_prob_f(:,:,1)); % all transitions for year 1. For years 2,...,200 same
    b.transprob_male = squeeze(baseline.trans_prob_m(:,:,1)); % all transitions for year 1. For years 2,...,200 same
    
    input{2} = 1;
    input{18}=@mort_func_two_rrs;
    
    for i = 1:num_samps
        i
        if i==1
            input{4}=true;
            else
            input{4}=false;
        end
        
        c = athena_main_mod_initbyyear('two_p_ind_prev_mat',input{:});
        
        % saving transition probabilities
          ix=6; % number of years before policy
        for fsy=1:2
            % cessation
            sa_output.cessprob_f_12{fsy}(:,i) = squeeze(c.trans_prob_f_saved{2}(:,12,fsy+ix));
            sa_output.cessprob_m_12{fsy}(:,i) = squeeze(c.trans_prob_m_saved{2}(:,12,fsy+ix));
           
            sa_output.cessprob_f_42{fsy}(:,i) = squeeze(c.trans_prob_f_saved{2}(:,42,fsy+ix));
            sa_output.cessprob_m_42{fsy}(:,i) = squeeze(c.trans_prob_m_saved{2}(:,42,fsy+ix));
            
            sa_output.cessprob_f_44{fsy}(:,i) = squeeze(c.trans_prob_f_saved{2}(:,44,fsy+ix));
            sa_output.cessprob_m_44{fsy}(:,i) = squeeze(c.trans_prob_m_saved{2}(:,44,fsy+ix));
            
            sa_output.cessprob_f_45{fsy}(:,i) = squeeze(c.trans_prob_f_saved{2}(:,45,fsy+ix));
            sa_output.cessprob_m_45{fsy}(:,i) = squeeze(c.trans_prob_m_saved{2}(:,45,fsy+ix));
            
    
            sa_output.cessprob_f_72{fsy}(:,i) = squeeze(c.trans_prob_f_saved{2}(:,72,fsy+ix));
            sa_output.cessprob_m_72{fsy}(:,i) = squeeze(c.trans_prob_m_saved{2}(:,72,fsy+ix));
          
          % switching
        sa_output.switchprob_f_15{fsy}(:,i) = squeeze(c.trans_prob_f_saved{2}(:,15,fsy+ix));
        sa_output.switchprob_m_15{fsy}(:,i) = squeeze(c.trans_prob_m_saved{2}(:,15,fsy+ix));
        
        sa_output.switchprob_f_69{fsy}(:,i) = squeeze(c.trans_prob_f_saved{2}(:,69,fsy+ix));
        sa_output.switchprob_m_69{fsy}(:,i) = squeeze(c.trans_prob_m_saved{2}(:,69,fsy+ix));
            
            % Initiation
            sa_output.initprob_f_2{fsy}(:,i) = squeeze(c.trans_prob_f_saved{2}(:,2,fsy+ix));
            sa_output.initprob_m_2{fsy}(:,i) = squeeze(c.trans_prob_m_saved{2}(:,2,fsy+ix));
            
            sa_output.initprob_f_4{fsy}(:,i) = squeeze(c.trans_prob_f_saved{2}(:,4,fsy+ix));
            sa_output.initprob_m_4{fsy}(:,i) = squeeze(c.trans_prob_m_saved{2}(:,4,fsy+ix));
            
            sa_output.initprob_f_5{fsy}(:,i) = squeeze(c.trans_prob_f_saved{2}(:,5,fsy+ix));
            sa_output.initprob_m_5{fsy}(:,i) = squeeze(c.trans_prob_m_saved{2}(:,5,fsy+ix));
        end
        
        for zzz=2:1+input{2}
            %zzz=input{2}+1;  %how many policies
            sa_output.adult_prev_c1{zzz-1}(:,i)=c.adult_prev_c1{zzz}(:,1:input{12});
            sa_output.adult_prev_c2{zzz-1}(:,i)=c.adult_prev_c2{zzz}(:,1:input{12});
            sa_output.adult_prev_cc{zzz-1}(:,i)=c.adult_prev_cc{zzz}(:,1:input{12});
            
            sa_output_aux.adult_prev_n1{zzz-1}(:,i)=c.adult_prev_n1{zzz}(:,1:input{12});
            sa_output_aux.adult_prev_n2{zzz-1}(:,i)=c.adult_prev_n2{zzz}(:,1:input{12});
        sa_output_aux.adult_prev_f1{zzz-1}(:,i)=c.adult_prev_f1{zzz}(:,1:input{12});
        sa_output_aux.adult_prev_f2{zzz-1}(:,i)=c.adult_prev_f2{zzz}(:,1:input{12});
        
        sa_output_aux.adult_pop_c1{zzz-1}(:,i)=c.adult_pop_c1{zzz}(:,1:input{12});
        sa_output_aux.adult_pop_c2{zzz-1}(:,i)=c.adult_pop_c2{zzz}(:,1:input{12});
        sa_output_aux.adult_pop_cc{zzz-1}(:,i)=c.adult_pop_cc{zzz}(:,1:input{12});
        
        sa_output_aux.adult_pop_n1{zzz-1}(:,i)=c.adult_pop_n1{zzz}(:,1:input{12});
        sa_output_aux.adult_pop_n2{zzz-1}(:,i)=c.adult_pop_n2{zzz}(:,1:input{12});
        sa_output_aux.adult_pop_f1{zzz-1}(:,i)=c.adult_pop_f1{zzz}(:,1:input{12});
        sa_output_aux.adult_pop_f2{zzz-1}(:,i)=c.adult_pop_f2{zzz}(:,1:input{12});
        
        % current users of product 1 regardless of use of product 2
        sa_output_aux.smokers{zzz-1}(:,:,i) = c.smokers{zzz}(1:input{12},:);
        
        sa_output.prev1824_c1{zzz-1}(:,i)=c.prev_cn{zzz,1}(:,1:input{12})+c.prev_cc{zzz,1}(:,1:input{12})+c.prev_cf{zzz,1}(:,1:input{12});
        sa_output.prev1824_c2{zzz-1}(:,i)=c.prev_nc{zzz,1}(:,1:input{12})+c.prev_cc{zzz,1}(:,1:input{12})+c.prev_fc{zzz,1}(:,1:input{12});
        sa_output.prev1824_cc{zzz-1}(:,i)=c.prev_cc{zzz,1}(:,1:input{12});
        
        sa_output.prev2544_c1{zzz-1}(:,i)=c.prev_cn{zzz,2}(:,1:input{12})+c.prev_cc{zzz,2}(:,1:input{12})+c.prev_cf{zzz,2}(:,1:input{12});
        sa_output.prev2544_c2{zzz-1}(:,i)=c.prev_nc{zzz,2}(:,1:input{12})+c.prev_cc{zzz,2}(:,1:input{12})+c.prev_fc{zzz,2}(:,1:input{12});
        sa_output.prev2544_cc{zzz-1}(:,i)=c.prev_cc{zzz,2}(:,1:input{12});
        
        sa_output.prev4564_c1{zzz-1}(:,i)=c.prev_cn{zzz,3}(:,1:input{12})+c.prev_cc{zzz,3}(:,1:input{12})+c.prev_cf{zzz,3}(:,1:input{12});
        sa_output.prev4564_c2{zzz-1}(:,i)=c.prev_nc{zzz,3}(:,1:input{12})+c.prev_cc{zzz,3}(:,1:input{12})+c.prev_fc{zzz,3}(:,1:input{12});
        sa_output.prev4564_cc{zzz-1}(:,i)=c.prev_cc{zzz,3}(:,1:input{12});
        
        sa_output.prev65100_c1{zzz-1}(:,i)=c.prev_cn{zzz,4}(:,1:input{12})+c.prev_cc{zzz,4}(:,1:input{12})+c.prev_cf{zzz,4}(:,1:input{12});
        sa_output.prev65100_c2{zzz-1}(:,i)=c.prev_nc{zzz,4}(:,1:input{12})+c.prev_cc{zzz,4}(:,1:input{12})+c.prev_fc{zzz,4}(:,1:input{12});
        sa_output.prev65100_cc{zzz-1}(:,i)=c.prev_cc{zzz,4}(:,1:input{12});
        
        sa_output.new_cessp1{zzz-1}(:,i)=[sum(c.new_cess_p1_f{zzz}(:,1:input{12})+c.new_cess_p1_m{zzz}(:,1:input{12}))]'; % product 1
        sa_output.new_cessp2{zzz-1}(:,i)=[sum(c.new_cess_p2_f{zzz}(:,1:input{12})+c.new_cess_p2_m{zzz}(:,1:input{12}))]'; % product 2
        sa_output.new_initp1{zzz-1}(:,i)=[sum(c.new_init_p1_f{zzz}(:,1:input{12})+c.new_init_p1_m{zzz}(:,1:input{12}))]'; % additional results for new smoking initiates
        sa_output.new_initp2{zzz-1}(:,i)=[sum(c.new_init_p2_f{zzz}(:,1:input{12})+c.new_init_p2_m{zzz}(:,1:input{12}))]'; % additional results for new non combusted initiates
        sa_output.pad{zzz-1}(:,i)=[sum(c.pad_vec_f{zzz}(:,1:input{12})+c.pad_vec_m{zzz}(:,1:input{12}))]';
        sa_output.padP1{zzz-1}(:,i)=[sum(c.padP1_vec_f{zzz}(:,1:input{12})+c.padP1_vec_m{zzz}(:,1:input{12}))]';
        sa_output.padP2{zzz-1}(:,i)=[sum(c.padP2_vec_f{zzz}(:,1:input{12})+c.padP2_vec_m{zzz}(:,1:input{12}))]';
        sa_output.padDUAL{zzz-1}(:,i)=[sum(c.padDUAL_vec_f{zzz}(:,1:input{12})+c.padDUAL_vec_m{zzz}(:,1:input{12}))]';
        
        
        sa_output.new_cessp1E{zzz-1}(:,i)=[sum(c.new_cess_p1E_f{zzz}(:,1:input{12})+c.new_cess_p1E_m{zzz}(:,1:input{12}))]'; % NEW product 1 exclusive 
        sa_output.new_cessp1Swtch{zzz-1}(:,i)=[sum(c.new_cess_p1Swtch_f{zzz}(:,1:input{12})+c.new_cess_p1Swtch_m{zzz}(:,1:input{12}))]'; % NEW product 1 exclusive 
        
        sa_output_aux.ave_adult18to100_cess_p1{zzz-1}(:, i) =c.ave_ad_cess_p1{zzz}(:,:);
        sa_output_aux.ave_adult18to100_cess_p2{zzz-1}(:,i) = c.ave_ad_cess_p2{zzz}(:,:);
        %sa_output_aux.ave_adult18to100_init_p1{zzz-1}(:,i) = c.ave_ad_init_p1{zzz}(:,:);
        %sa_output_aux.ave_adult18to100_init_p2{zzz-1}(:,i) = c.ave_ad_init_p2{zzz}(:,:);
        %sa_output_aux.ave_youth12to17_cess_p1{zzz-1}(:,i) = c.ave_y_cess_p1{zzz}(:,:);
        %sa_output_aux.ave_youth12to17_cess_p2{zzz-1}(:,i) = c.ave_y_cess_p2{zzz}(:,:);
        sa_output_aux.ave_youth12to17_init_p1{zzz-1}(:,i) = c.ave_y_init_p1{zzz}(:,:);
        sa_output_aux.ave_youth12to17_init_p2{zzz-1}(:,i) = c.ave_y_init_p2{zzz}(:,:);
        
        sa_output.lifeyears{zzz-1}(:,i)=c.lifeyears{zzz}(2:input{12}+1,:);
        sa_output.ad_morb_qaly_lost{zzz-1}(:,i)=c.ad_morb_qaly_lost{zzz}(2:input{12}+1,:);
        sa_output_aux.adult_qaly{zzz-1}(:,i)=c.adult_qaly{zzz}(2:input{12}+1,:);
        sa_output_aux.age_deaths35to84{zzz-1}(:,i)=c.age_deaths{zzz}(1:input{12},:);
        
        % For validation
        sa_output.pop_tot{zzz-1}(:,i)=c.pop_tot{zzz}(:,1:input{12});
        sa_output.death_f{zzz-1}(:,i)=sum(c.death_vec_f{zzz}(:,1:input{12}),1);
        sa_output.death_m{zzz-1}(:,i)=sum(c.death_vec_m{zzz}(:,1:input{12}),1);
        sa_output.births{zzz-1}(:,i)=c.births{zzz}(:,1:input{12});
        sa_output.pop_female1534{zzz-1}(:,i)=c.pop_female1534{zzz}(:,1:input{12});
        sa_output.pop1824{zzz-1}(:,i)=c.sub_pop{zzz,1}(:,1:input{12}); %c.sub_pop_f{zzz,1}(:,1:input{12}) + c.sub_pop_m{zzz,1}(:,1:input{12});
        sa_output.pop2544{zzz-1}(:,i)=c.sub_pop{zzz,2}(:,1:input{12}); %c.sub_pop_f{zzz,2}(:,1:input{12}) + c.sub_pop_m{zzz,2}(:,1:input{12});
        sa_output.pop4564{zzz-1}(:,i)=c.sub_pop{zzz,3}(:,1:input{12}); %c.sub_pop_f{zzz,3}(:,1:input{12}) + c.sub_pop_m{zzz,3}(:,1:input{12});
        sa_output.pop65100{zzz-1}(:,i)=c.sub_pop{zzz,4}(:,1:input{12}); %c.sub_pop_f{zzz,4}(:,1:input{12}) + c.sub_pop_m{zzz,4}(:,1:input{12});
        
        end
        end
        
        sa_output.samps=samps';
        sa_output.prob_vals=prob_vals';
        sa_output.val_sets=val_sets';
        
        for zzz=2:1+input{2}
        sa_output.cum_dif_new_cessp1{zzz-1} = cumsum( bsxfun( @minus, sa_output.new_cessp1{zzz-1}, b.new_cessp1) );
        sa_output.cum_dif_new_cessp2{zzz-1} = cumsum( bsxfun( @minus, sa_output.new_cessp2{zzz-1}, b.new_cessp2) );
        sa_output.cum_dif_new_initp1{zzz-1} = cumsum( bsxfun( @minus, b.new_initp1, sa_output.new_initp1{zzz-1}) );
        sa_output.cum_dif_new_initp2{zzz-1} = cumsum( bsxfun( @minus, sa_output.new_initp2{zzz-1},b.new_initp2) );
        sa_output.cum_dif_pad{zzz-1} = cumsum( bsxfun( @minus, sa_output.pad{zzz-1}, b.pad ) );
        sa_output.cum_dif_lifeyears{zzz-1} = cumsum( bsxfun( @minus, sa_output.lifeyears{zzz-1}, b.lifeyears ) );
        sa_output.cum_dif_ad_morb_qaly_lost{zzz-1} = cumsum( bsxfun( @minus, sa_output.ad_morb_qaly_lost{zzz-1}, b.ad_morb_qaly_lost) );
        sa_output.adult_prev_ctot{zzz-1}=sa_output.adult_prev_c1{zzz-1}+sa_output.adult_prev_c2{zzz-1}-sa_output.adult_prev_cc{zzz-1};
        
        sa_output.cum_dif_new_cessp1E{zzz-1} = cumsum( bsxfun( @minus, sa_output.new_cessp1E{zzz-1}, b.new_cessp1E) ); % NEW product 1 exclusive 
        sa_output.cum_dif_new_cessp1Swtch{zzz-1} = cumsum( bsxfun( @minus, sa_output.new_cessp1Swtch{zzz-1}, b.new_cessp1Swtch) ); % NEW product 1 exclusive 
        
        sa_output.prev1824_ctot{zzz-1}=sa_output.prev1824_c1{zzz-1}+sa_output.prev1824_c2{zzz-1}-sa_output.prev1824_cc{zzz-1};
        sa_output.prev2544_ctot{zzz-1}=sa_output.prev2544_c1{zzz-1}+sa_output.prev2544_c2{zzz-1}-sa_output.prev2544_cc{zzz-1};
        sa_output.prev4564_ctot{zzz-1}=sa_output.prev4564_c1{zzz-1}+sa_output.prev4564_c2{zzz-1}-sa_output.prev4564_cc{zzz-1};
        sa_output.prev65100_ctot{zzz-1}=sa_output.prev65100_c1{zzz-1}+sa_output.prev65100_c2{zzz-1}-sa_output.prev65100_cc{zzz-1};
        
        end
        
        sa_tables = add_tables(b, b_aux, sa_output, sa_output_aux);
        
        function [ c ] = wrapper_func(c, policy_ind)
        % Calls function handle referenced by enclosing function.
        c.samps=samps(i,:);
        c=policy_func(c,policy_ind);
        end
        
        end
        
        function [samps, prob_vals, val_sets] = do_samps(num_samps, vals, pctiles)
        val_set_num=size(vals,3);
        samps=[]; prob_vals=[]; val_sets=[];
        for val_set=1:val_set_num
        [ s, pv, vs ] = lhs_lin_interp_func( num_samps/val_set_num, vals(:,:,val_set), pctiles );
        samps=[samps;s];
        prob_vals=[prob_vals;pv];
        val_sets=[val_sets;vs*val_set];
        end
        end
        
        
                