%{
    Nicotine Product Standard model, INCLUDING PARAMETER UNCERTAINTY via Latin hypercube sampling
    This code is to run the model and export outputs to an excel template.
    Last updated in May 2023.
    
    The time to run the model for 7000 simulations may vary from computer to computer.
    Using a scientific laptop, it takes about from 3 hours to 4.5 hours. 
    Baseline Year: 2021
    Nicotine Product standard implementation: 2027

How to run the model:

To run the main scenario (scenario 1):
   1) Uncomment the code below and inside  "Scenario 1 to run the main model"
   2) Comment the codes below and inside  "Scenario 2 to run the menthol
   adjusted model"
   3) Then run the whole codes (all sections included here)

To run the menthol adjusted model (scenario 2):
   1) Comment the code below and inside  "Scenario 1 to run the main model"
   2) Uncomment the codes below and inside  "Scenario 2 to run the menthol
   adjusted model"
   3) Then run the whole codes (all sections included here)

%}
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%clear; close all; clc
tStart = tic;
% sample inputs
% 7 is the number of experts, 1000 is the number of samples for each expert input
% it will produced 1000 simulations for each expert, 7000 simulations total.
nExperts =7;  % number of experts, do not change this.
nSimulations =1000; % number of simulations for each expert, 1000 is used since the model output is almost stable after 7000 simulations

num_samps =nExperts*nSimulations;  % total samples of the input; will produce 7000 simulations for the policy

Year = (2021:2100)';

%===== Scenario 1 to run the main model======

%  %{
% to run the model -main model
[sa_tables, sa_output, b ] = sa_main_mod_2021_NicotineModel( num_samps );

tElapsed = toc(tStart);
disp(['Elapsed time in min: ',num2str(tElapsed/60)]);

% to save the ouput data
save MATLAB_NicotineModelResults_2021Baseline2027Policy_Main.mat b num_samps sa_output sa_tables tElapsed;

% to export model outputs to an Excel template. The name
% should be the same as the template to be used for output
filename = "NicotineModelResults_EmptyTemplate.xlsx";



 % %}

%===== End of Scenario 1 to run the main model======


%===== Scenario 2 to run the menthol adjusted model======
  %{
% to run the  model - menthol adjusted model
[sa_tables, sa_output, b ] = sa_main_mod_2021_NicotineModelMentholAdjusted( num_samps );

tElapsed = toc(tStart);
disp(['Elapsed time in min: ',num2str(tElapsed/60)]);

% to save the ouputs data
save MATLAB_NicotineModelResults_2021Baseline2027Policy_MentholMean.mat b num_samps sa_output sa_tables tElapsed;
% to export model outputs to an Excel template. The name
% should be the same as the template to be used for output

filename = "NicotineModelMentholAdjustedResults_EmptyTemplate.xlsx";

 %}

%=====  End of Scenario 2 to run the menthol adjusted model======


%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% ------------------------------------------------------------------------
% % From  Census
load('ValidationData\deaths2015-2060.mat');


%  From 2017 National Population Projections
load('ValidationData\deaths2017-2060.mat');
load('ValidationData\pop2017-2060.mat');


% -------------------------------------------------------------------------
% FIGURE: Projected population
aux = prctile(sa_output.pop_tot{1},[5,50,95],2);
prev1.p5 = aux(:,1);
prev1.median = aux(:,2);
prev1.p95 = aux(:,3);

figure();
plot(2021:2100,b.pop_tot/1e6,'b',...
2021:2100,prev1.p95/1e6,'-.k', ...
2021:2100,prev1.median/1e6,'-k', ...
2021:2100,prev1.p5/1e6,'--k',...
2021:2060,pop2017_2060(5:end)/1e6,'or','Markersize',3,'LineWidth',3); set(gca,'FontSize',18)
grid on
axis([2021 2100 300 480])
xlabel('Years')
ylabel('Projected Population (Millions)')
legend('Baseline','Policy: 95th percentile','Policy: Median','Policy: 5th percentile',...
'Census Projection','Location','southeast');

Baseline = b.pop_tot;
Policy5thPercentile = prev1.p5;
PolicyMedian = prev1.median;
Policy95thPercentile = prev1.p95;
Census2010 = NaN(80,1);
Census2010(1:40) = pop2017_2060(5:end);
T = table(Year, Baseline, Policy5thPercentile, PolicyMedian, Policy95thPercentile, Census2010);
writetable(T,filename,'Sheet',"PopulationProjections",'Range','A3:F82','WriteVariableNames',0);

% -----------------------------------------------------------
% FIGURE: Deaths
aux1 = prctile(sa_output.death_f{1},[5,50,95],2);
aux2 = prctile(sa_output.death_m{1},[5,50,95],2);
aux = aux1 + aux2;
prev1.p5 = aux(:,1);
prev1.median = aux(:,2);
prev1.p95 = aux(:,3);
%
figure();
plot(2021:2100,(b.death_f+b.death_m)/1e6,'b',...
2021:2100,prev1.p95/1e6,'-.k', ...
2021:2100,prev1.median/1e6,'-k', ...
2021:2100,prev1.p5/1e6,'--k',...
2021:2060,deaths201560(7:end)/1e6,'or','Markersize',3,'LineWidth',3); set(gca,'FontSize',18)
grid on
axis([2021 2100 2 5]) %min(min(c.adult_prev_c1{1:2})) max(max(c.adult_prev_c1{1:2}))])
xlabel('Years')
ylabel('Projected Deaths (Millions)')
legend('Baseline','Policy: 95th percentile','Policy: Median','Policy: 5th percentile',...
'Census Projection','Location','southeast');

% Exporting to an Excel file:
Year = (2021:2100)';
Baseline = b.death_f+b.death_m;
Policy5thPercentile = prev1.p5;
PolicyMedian = prev1.median;
Policy95thPercentile = prev1.p95;
Census2010 = NaN(80,1);
Census2010(1:40) = deaths201560(7:end);
%Census2000 = NaN(86,1);
%Census2000(1:36) = deaths201550;
T = table(Year, Baseline, Policy5thPercentile, PolicyMedian, Policy95thPercentile, Census2010);
writetable(T,filename,'Sheet',"DeathsProjections",'Range','A3:F82','WriteVariableNames',0);


% ------------------ Panel(a):cigarette ----------------------
aux = prctile(sa_output.adult_prev_c1{1},[5,50,95],2);
prev1.p5 = aux(:,1);
prev1.median = aux(:,2);
prev1.p95 = aux(:,3);

figure();
plot(2021:2100,b.adult_prev_c1,'b',...
2021:2100,prev1.p95,'-.k', ...
2021:2100,prev1.median,'-k', ...
2021:2100,prev1.p5,'--k','LineWidth',3); set(gca,'FontSize',18)
grid on
axis([2021 2100 0 0.18]) %min(min(c.adult_prev_c1{1:2})) max(max(c.adult_prev_c1{1:2}))])
xlabel('Years')
ylabel('Prevalence')
legend('Baseline','Policy: 95th percentile','Policy: Median','Policy: 5th percentile','Location','northeast');
title({'Projected Cigarette Smoking Prevalence', 'Among U.S. Adults (18+)'})

% Exporting to an Excel file:
Year = (2021:2100)';
Baseline = b.adult_prev_c1;
Policy5thPercentile = prev1.p5;
PolicyMedian = prev1.median;
Policy95thPercentile = prev1.p95;
T = table(Year, Baseline, Policy5thPercentile, PolicyMedian, Policy95thPercentile);
writetable(T,filename,'Sheet',"Cigarette Smoking Prevalence",'Range','A4:E83','WriteVariableNames',0);

% ------------------ Panel(b): non-combusted ----------------------
aux = prctile(sa_output.adult_prev_c2{1},[5,50,95],2);
prev2.p5 = aux(:,1);
prev2.median = aux(:,2);
prev2.p95 = aux(:,3);

figure();
plot(2021:2100,b.adult_prev_c2,'b',...
2021:2100,prev2.p95,'-.k', ...
2021:2100,prev2.median,'-k', ...
2021:2100,prev2.p5,'--k','LineWidth',3); set(gca,'FontSize',17)
grid on
axis([2021 2100 0 0.2])
xlabel('Years')
ylabel('Prevalence')
legend('Baseline','Policy: 95th percentile','Policy: Median','Policy: 5th percentile','Location','northeast');
title({'Projected Non-combusted use Prevalence', 'Among U.S. Adults (18+)'})

% Exporting to an Excel file:
Baseline = b.adult_prev_c2;
Policy5thPercentile = prev2.p5;
PolicyMedian = prev2.median;
Policy95thPercentile = prev2.p95;
T = table(Year, Baseline, Policy5thPercentile, PolicyMedian, Policy95thPercentile);
writetable(T,filename,'Sheet',"Non-Combusted Use Prevalence",'Range','A4:E83','WriteVariableNames',0);

% ------------------ Panel(c): dual users  ----------------------
aux = prctile(sa_output.adult_prev_cc{1},[5,50,95],2);
prevcc.p5 = aux(:,1);
prevcc.median = aux(:,2);
prevcc.p95 = aux(:,3);

figure();
plot(2021:2100,b.adult_prev_cc,'b',...
2021:2100,prevcc.p95,'-.k', ...
2021:2100,prevcc.median,'-k', ...
2021:2100,prevcc.p5,'--k','LineWidth',3); set(gca,'FontSize',17)
grid on
axis([2021 2100 0 0.2]) %18])
xlabel('Years')
ylabel('Prevalence')
legend('Baseline','Policy: 95th percentile','Policy: Median','Policy: 5th percentile','Location','northeast');
title({'Projected dual use Prevalence', 'Among U.S. Adults (18+)'})

% Exporting to an Excel file:
Baseline = b.adult_prev_cc;
Policy5thPercentile = prevcc.p5;
PolicyMedian = prevcc.median;
Policy95thPercentile = prevcc.p95;
T = table(Year, Baseline, Policy5thPercentile, PolicyMedian, Policy95thPercentile);
writetable(T,filename,'Sheet',"Dual Use Prevalence",'Range','A4:E83','WriteVariableNames',0);

% ------------------ Panel(d): any tobacco use  ----------------------
aux = prctile(sa_output.adult_prev_ctot{1},[5,50,95],2); %[5,50,95]
prevctot.p5 = aux(:,1);
prevctot.median = aux(:,2);
prevctot.p95 = aux(:,3);

figure();
plot(2021:2100,b.adult_prev_ctot,'b',...
2021:2100,prevctot.p95,'-.k', ...
2021:2100,prevctot.median,'-k', ...
2021:2100,prevctot.p5,'--k','LineWidth',3); set(gca,'FontSize',17)
grid on
axis([2021 2100 0 0.25]);
xlabel('Years')
ylabel('Prevalence')
legend('Baseline','Policy: 95th percentile','Policy: Median','Policy: 5th percentile','Location','northeast');
title({'Projected any tobacco use Prevalence', 'Among U.S. Adults (18+)'})

% Exporting to an Excel file:
Baseline = b.adult_prev_ctot;
Policy5thPercentile = prevctot.p5;
PolicyMedian = prevctot.median;
Policy95thPercentile = prevctot.p95;
T = table(Year, Baseline, Policy5thPercentile, PolicyMedian, Policy95thPercentile);
writetable(T,filename,'Sheet',"Any Tobacco Use Prevalence",'Range','A4:E83','WriteVariableNames',0);


%% outcome metrics
% (1) Cumulative new smoking initiates avoided
% (2) Cumulative number of people who quit smoking as a result of the policy
% (3) Cumulative tobacco attributable deaths avoided
% (4) Cumulative life years gained
% (5) Cumulative QALYs gained
%
% For years: 2025; 2030; 2040; 2050; 2060; 2070; 2080; 2090; 2100

% -------------------------------------------------------------------------
% (1) Cumulative new smoking initiates avoided
initavoided = prctile(sa_output.cum_dif_new_initp1{1},[5,50,95],2);
disp('Cumulative new init. avoided (millions): median, 5%P, 95%P ');
years_aux = [2025; 2030; 2040; 2050; 2060; 2070; 2080; 2090; 2100];
ind = years_aux - 2021 + 1;
disp(table(years_aux, initavoided(ind,[1,2,3])/1e+6, ...
'VariableNames', {'Year' 'P5thP_Median_95thP'}));

% Exporting to an Excel file:
InitiatesAvoided_5thPercentile = initavoided(:,1);
InitiatesAvoided_median = initavoided(:,2);
InitiatesAvoided_95thPercentile = initavoided(:,3);
T = table(Year, InitiatesAvoided_5thPercentile, InitiatesAvoided_median, InitiatesAvoided_95thPercentile);
writetable(T,filename,'Sheet',"CumSmokingInitiatesAvoided",'Range','A4:E83','WriteVariableNames',0);

% -------------------------------------------------------------------------
% (2) Cumulative number of people who quit smoking as a result of the policy
SmokingQuitters_baseline =b.new_cessp1;
SmokingQuitters_policy=prctile(sa_output.new_cessp1{1},[5,50,95],2);
quitters = prctile(sa_output.cum_dif_new_cessp1{1},[5,50,95],2);
%disp('Cumulative number of quitters (millions): median, 5%P, 95%P ');
%disp(table([2020; 2024; 2029; 2034], quitters([4,8,13,18],[1,2,3])/1e+6, ...
%    'VariableNames', {'Year' 'P5thP_Median_95thP'}));

% Exporting to an Excel file:
Quitters_5thPercentile = quitters(:,1);
Quitters_median = quitters(:,2);
Quitters_95thPercentile = quitters(:,3);
T = table(Year, SmokingQuitters_baseline, SmokingQuitters_policy, Quitters_5thPercentile, Quitters_median, Quitters_95thPercentile);
writetable(T,filename,'Sheet',"CumSmokingQuitters",'Range','A3:H82','WriteVariableNames',0);
% -------------------------------------------------------------------------
% (3) Cumulative tobacco attributable deaths avoided = baseline - policy
ad.base = b.pad; %%sum(c.padP1_vec_f{1} + c.padP1_vec_m{1},1)';
ad.poli = sa_output.pad{1}; %sum(c.padP1_vec_f{2} + c.padP1_vec_m{2},1)';
ad.avoided = prctile(cumsum(bsxfun(@minus,ad.base,ad.poli)),[5,50,95],2); %[5,50,95]
CumAttrDeathAvoided_5thPercentile = ad.avoided(:,1); % 5th Percentile
CumAttrDeathAvoided_median = ad.avoided(:,2); % 50th Percentile
CumAttrDeathAvoided_95thPercentile = ad.avoided(:,3);   % 95th Percentile

T = table(Year, CumAttrDeathAvoided_5thPercentile, CumAttrDeathAvoided_median, CumAttrDeathAvoided_95thPercentile);
writetable(T,filename,'Sheet',"CumTobaccoAttrDeathsAvoided",'Range','A3:D82','WriteVariableNames',0);

cumTADsAvoided =cumsum(bsxfun(@minus,ad.base,ad.poli));
cumTADsAvoided_3Yminus =zeros(size(cumTADsAvoided));
cumTADsAvoided_nonNegative =max(cumTADsAvoided, 0);  % removing negtive values
%Following Apelberg et al. (2018), to be conservative, we excluded the first 3 years after the implementation of 
%  the policy from our cumulative estimates.
cumTADsAvoided_3Yminus(9:end,:) =cumTADsAvoided_nonNegative(9:end,:)-cumTADsAvoided_nonNegative(9,:); 
% row 9 is 2029 which is 3rd year of the policy
cumTADsAvoided_3Yminus_prcnt= prctile(cumTADsAvoided_3Yminus,[5,50,95],2); %[5,50,95]

cumTADsAvoided_3Yminus_prcnt5th = cumTADsAvoided_3Yminus_prcnt(:,1); % 5th Percentile
cumTADsAvoided_3Yminus_prcntMedian= cumTADsAvoided_3Yminus_prcnt(:,2); % 50th Percentile
cumTADsAvoided_3Yminus_prcnt95th = cumTADsAvoided_3Yminus_prcnt(:,3);   % 95th Percentile

T = table(cumTADsAvoided_3Yminus_prcnt5th,cumTADsAvoided_3Yminus_prcntMedian,cumTADsAvoided_3Yminus_prcnt95th);
writetable(T,filename,'Sheet',"CumTobaccoAttrDeathsAvoided",'Range','F3:H82','WriteVariableNames',0);

% -------------------------------------------------------------------------
% (4) Cumulative life years gained = policy - baseline
% cumlife = prctile(sa_output.cum_dif_lifeyears{1},[5,50,95],2);

lifeyears.base = b.lifeyears; %%sum(c.padP1_vec_f{1} + c.padP1_vec_m{1},1)';
lifeyears.poli = sa_output.lifeyears{1}; %sum(c.padP1_vec_f{2} + c.padP1_vec_m{2},1)';
lifeyears.avoided = prctile(cumsum( bsxfun( @minus,lifeyears.poli,lifeyears.base ) ),[5,50,95],2); %[5,50,95]

% Exporting to an Excel file:
cumLifeGained_5thPrcnt =lifeyears.avoided(:,1); % if the difference is negative, replace it by 0
cumLifeGained_median = lifeyears.avoided(:,2);
cumLifeGained_95thPrcnt = lifeyears.avoided(:,3);
T = table(Year, cumLifeGained_5thPrcnt, cumLifeGained_median, cumLifeGained_95thPrcnt);
writetable(T,filename,'Sheet',"CumLifeYearsGained",'Range','A3:D82','WriteVariableNames',0);


cumLifeGained =cumsum(bsxfun( @minus,lifeyears.poli,lifeyears.base ));
cumLifeGained_3Yminus =zeros(size(cumLifeGained));
cumLifeGained_nonNegative =max(cumLifeGained, 0);  % removing negtive values
%Following Apelberg et al. (2018), to be conservative, we excluded the first 3 years after the implementation of 
%  the policy from our cumulative estimates.
cumLifeGained_3Yminus(9:end,:) =cumLifeGained_nonNegative(9:end,:)-cumLifeGained_nonNegative(9,:); % row 9 is 2029 which is 3rd year of the policy
cumLifeGained_3Yminus_prcnt= prctile(cumLifeGained_3Yminus,[5,50,95],2); %[5,50,95]

cumLifeGained_3Yminus_prcnt5th = cumLifeGained_3Yminus_prcnt(:,1); % 5th Percentile
cumLifeGained_3Yminus_prcntMedian= cumLifeGained_3Yminus_prcnt(:,2); % 50th Percentile
cumLifeGained_3Yminus_prcnt95th = cumLifeGained_3Yminus_prcnt(:,3);   % 95th Percentile

T = table(cumLifeGained_3Yminus_prcnt5th,cumLifeGained_3Yminus_prcntMedian,cumLifeGained_3Yminus_prcnt95th);
writetable(T,filename,'Sheet',"CumLifeYearsGained",'Range','F3:H82','WriteVariableNames',0);

% -------------------------------------------------------------------------
% (5) Cumulative QALYs gained


ad_morb_qaly_lost.base = b.ad_morb_qaly_lost; %%sum(c.padP1_vec_f{1} + c.padP1_vec_m{1},1)';
ad_morb_qaly_lost.poli = sa_output.ad_morb_qaly_lost{1}; %sum(c.padP1_vec_f{2} + c.padP1_vec_m{2},1)';
cum_ad_morb_QALYLost = prctile(cumsum( bsxfun( @minus,ad_morb_qaly_lost.base, ad_morb_qaly_lost.poli ) ),[5,50,95],2); %[5,50,95]

% Exporting to an Excel file:
QALYLost_5thPrcnt =cum_ad_morb_QALYLost(:,1); % if the difference is negative, replace it by 0
QALYLost_median = cum_ad_morb_QALYLost(:,2);
QALYLost_95thPrcnt = cum_ad_morb_QALYLost(:,3);
T = table(Year, QALYLost_5thPrcnt, QALYLost_median, QALYLost_95thPrcnt);
writetable(T,filename,'Sheet',"CumQALYGained",'Range','A3:D82','WriteVariableNames',0);

cumQALYLost=cumsum(bsxfun( @minus,ad_morb_qaly_lost.base, ad_morb_qaly_lost.poli ));
cumQALYLost_3Yminus =zeros(size(cumQALYLost));
cumQALYLost_nonNegative =max(cumQALYLost, 0);  % removing negtive values
% Following Apelberg et al.(2018), to be conservative, we excluded the first 3 years after the implementation of 
% the policy from our cumulative estimates.
cumQALYLost_3Yminus(9:end,:) =cumQALYLost_nonNegative(9:end,:)-cumQALYLost_nonNegative(9,:); % row 9 is 2029 which is 3rd year of the policy
cumQALYLost_3Yminus_prcnt= prctile(cumQALYLost_3Yminus,[5,50,95],2); %[5,50,95]

cumQALYLost_3Yminus_prcnt5th = cumQALYLost_3Yminus_prcnt(:,1); % 5th Percentile
cumQALYLost_3Yminus_prcntMedian= cumQALYLost_3Yminus_prcnt(:,2); % 50th Percentile
cumQALYLost_3Yminus_prcnt95th = cumQALYLost_3Yminus_prcnt(:,3);   % 95th Percentile

T = table(cumQALYLost_3Yminus_prcnt5th,cumQALYLost_3Yminus_prcntMedian,cumQALYLost_3Yminus_prcnt95th);
writetable(T,filename,'Sheet',"CumQALYGained",'Range','F3:H82','WriteVariableNames',0);


% -------------------------------------------------------------------------
%% Tobacco-Attributable Deaths: Other deaths.
% Dataa Sources:
% Direct Adult Smoking-Attributable Deaths, 2005-2009 - 2014 Surgeon General's Report https://www.cdc.gov/tobacco/sgr/50th-anniversary/index.htm
% Secondhand Smoke Deaths, 2005-2009 - 2014 Surgeon General's Report
% Perinatal Conditions Deaths, 2005-2009 - 2014 Surgeon General's Report
% Home Fires Deaths, 2012-2016 - 2019 NFPA Report https://www.nfpa.org/News-and-Research/Data-research-and-tools/US-Fire-Problem/Smoking-Materials
% Change in Tobacco-attributable Deaths by Year - Model Projections
%Note the following values can be updated using new data.

ScndHandSmoke=41280;   % Secondhand Smoke, 2005-2009
PrntlAll=1013;%Perinatal Conditions, 2005-2009  Perinatal Conditions All means - Prenatal Conditions and SIDS
%PerinatalCond =613;
%SIDS=400;
HomeFires=590; %Home Fires, 2012-2016
DirectAdult_TADs = 437400; % Direct Adult Smoking-Attributable Deaths, 2005-2009

% ------ Table: Cumulative Secondhand Smoke Attributable Deaths Avoided -------
ScndHandSmokeDeath_base = (ScndHandSmoke/DirectAdult_TADs).*b.padP1; %%sum(c.padP1_vec_f{1} + c.padP1_vec_m{1},1)';
ScndHandSmokeDeath_poli = (ScndHandSmoke/DirectAdult_TADs).*sa_output.padP1{1}; %sum(c.padP1_vec_f{2} + c.padP1_vec_m{2},1)';
cumScndHandSmokeDeathAvoided =cumsum(bsxfun(@minus,ScndHandSmokeDeath_base,ScndHandSmokeDeath_poli ));
cumScndHandSmokeDeathAvoided_3Yminus =zeros(size(cumScndHandSmokeDeathAvoided));
cumScndHandSmokeDeathAvoided_nonNegative =max(cumScndHandSmokeDeathAvoided, 0);  % removing negtive values
cumScndHandSmokeDeathAvoided_3Yminus(9:end,:) =cumScndHandSmokeDeathAvoided_nonNegative(9:end,:)-cumScndHandSmokeDeathAvoided_nonNegative(9,:); % row 8 is 2028 which is 3rd year of the policy

% Exctracting 5th Percentile, 50th Percentile, 95th Percentile
cumScndHandSmokeDeathAvoided_3Yminus_prcnt= prctile(cumScndHandSmokeDeathAvoided_3Yminus,[5,50,95],2); %[5,50,95]

% Exporting to an Excel file:
Year = (2021:2100)';
T = table(Year,ScndHandSmokeDeath_base, cumScndHandSmokeDeathAvoided_3Yminus_prcnt);
writetable(T,filename,'Sheet',"MortalityReducedOtherDeaths",'Range','A4:E83','WriteVariableNames',0);

% ----- Table: Cumulative Perinatal Deaths Avoided --------------
PrntlAllDeath_base = (PrntlAll/DirectAdult_TADs).*b.padP1; %%sum(c.padP1_vec_f{1} + c.padP1_vec_m{1},1)';
PrntlAllDeath_poli = (PrntlAll/DirectAdult_TADs).*sa_output.padP1{1}; %sum(c.padP1_vec_f{2} + c.padP1_vec_m{2},1)';
cumPrntlAllDeathAvoided =cumsum(bsxfun(@minus,PrntlAllDeath_base,PrntlAllDeath_poli ));
cumPrntlAllDeathAvoided_3Yminus =zeros(size(cumPrntlAllDeathAvoided));
cumPrntlAllDeathAvoided_nonNegative =max(cumPrntlAllDeathAvoided, 0);  % removing negtive values
cumPrntlAllDeathAvoided_3Yminus(9:end,:) =cumPrntlAllDeathAvoided_nonNegative(9:end,:)-cumPrntlAllDeathAvoided_nonNegative(9,:); % row 8 is 2028 which is 3rd year of the policy

cumPrntlAllDeathAvoided_3Yminus_prcnt= prctile(cumPrntlAllDeathAvoided_3Yminus,[5,50,95],2); %[5,50,95] Percentile
% Exporting to an Excel file:
Year = (2021:2100)';
T = table(Year,PrntlAllDeath_base, cumPrntlAllDeathAvoided_3Yminus_prcnt);
writetable(T,filename,'Sheet',"MortalityReducedOtherDeaths",'Range','G4:K83','WriteVariableNames',0);

% ------ Table: Cumulative Smoking-related Fire Deaths Avoided---------
HomeFiresDeath_base = (HomeFires/DirectAdult_TADs).*b.padP1; %%sum(c.padP1_vec_f{1} + c.padP1_vec_m{1},1)';
HomeFiresDeath_poli = (HomeFires/DirectAdult_TADs).*sa_output.padP1{1}; %sum(c.padP1_vec_f{2} + c.padP1_vec_m{2},1)';
cumHomeFiresDeathAvoided =cumsum(bsxfun(@minus,HomeFiresDeath_base,HomeFiresDeath_poli ));
cumHomeFiresDeathAvoided_3Yminus =zeros(size(cumHomeFiresDeathAvoided));
cumHomeFiresDeathAvoided_nonNegative =max(cumHomeFiresDeathAvoided, 0);  % removing negtive values
cumHomeFiresDeathAvoided_3Yminus(9:end,:) =cumHomeFiresDeathAvoided_nonNegative(9:end,:)-cumHomeFiresDeathAvoided_nonNegative(9,:); % row 8 is 2028 which is 3rd year of the policy

cumHomeFiresDeathAvoided_3Yminus_prcnt= prctile(cumHomeFiresDeathAvoided_3Yminus,[5,50,95],2); %[5,50,95] Percentile
Year = (2021:2100)';
T = table(Year,HomeFiresDeath_base, cumHomeFiresDeathAvoided_3Yminus_prcnt);
writetable(T,filename,'Sheet',"MortalityReducedOtherDeaths",'Range','M4:Q83','WriteVariableNames',0);

% -----Table: Cumulative Non-Premium Cigar Attributable Deaths Avoided---

% ===== This was used for Cigar mortality post processing, but now it is not used ========
% prematuredeaths_allCigar = 9246;  % Nonnemaker et al. all cigar premature deaths
% %            % 80% reported non-premium cigars (i.e., little filtered cigars or non-premium cigarillos/large cigars)
% %      % and 20% reported premium cigars as their usual cigar type using a classification methodology described previously (Corey et al., 2014)
% NonPrmiumCigar = 0.8*prematuredeaths_allCigar;

% %
% %  cigarSmokingFlavor= 6616.8; %Cigar Smoking with Flavor Product Standard
% %   %     Mortality Reduction from Product Standard	780
% NonPrmiumCigarDeath_base = (NonPrmiumCigar/DirectAdult_TADs).*b.padP1; %%sum(c.padP1_vec_f{1} + c.padP1_vec_m{1},1)';
% NonPrmiumCigarDeath_poli = (NonPrmiumCigar/DirectAdult_TADs).*sa_output.padP1{1}; %sum(c.padP1_vec_f{2} + c.padP1_vec_m{2},1)';
% cumNonPrmiumCigarDeathAvoided =cumsum(bsxfun(@minus,NonPrmiumCigarDeath_base,NonPrmiumCigarDeath_poli ));
% cumNonPrmiumCigarDeathAvoided_3Yminus =zeros(size(cumNonPrmiumCigarDeathAvoided));
% cumNonPrmiumCigarDeathAvoided_nonNegative =max(cumNonPrmiumCigarDeathAvoided, 0);  % removing negtive values
% cumNonPrmiumCigarDeathAvoided_3Yminus(8:end,:) =cumNonPrmiumCigarDeathAvoided_nonNegative(8:end,:)-cumNonPrmiumCigarDeathAvoided_nonNegative(8,:); % row 8 is 2028 which is 3rd year of the policy

% cumNonPrmiumCigarDeathAvoided_3Yminus_prcnt= prctile(cumNonPrmiumCigarDeathAvoided_3Yminus,[5,50,95],2); %[5,50,95] Percentile
% % Exporting to an Excel file:
% Year = (2021:2100)';
% T = table(Year,NonPrmiumCigarDeath_base,cumNonPrmiumCigarDeathAvoided_3Yminus_prcnt);
% writetable(T,filename,'Sheet',"MortalityReducedOtherDeaths",'Range','S4:W83','WriteVariableNames',0);


% ---- Table: Cumulative Pipe Tobacco Attributable Deaths Avoided
PipeSmoking = 1095; %Pipe Smoking, 1991
PipeSmokingDeath_base = (PipeSmoking/DirectAdult_TADs).*b.padP1; %%sum(c.padP1_vec_f{1} + c.padP1_vec_m{1},1)';
PipeSmokingDeath_poli = (PipeSmoking/DirectAdult_TADs).*sa_output.padP1{1}; %sum(c.padP1_vec_f{2} + c.padP1_vec_m{2},1)';
cumPipeSmokingDeathAvoided =cumsum(bsxfun(@minus,PipeSmokingDeath_base,PipeSmokingDeath_poli ));
cumPipeSmokingDeathAvoided_3Yminus =zeros(size(cumPipeSmokingDeathAvoided));
cumPipeSmokingDeathAvoided_nonNegative =max(cumPipeSmokingDeathAvoided, 0);  % removing negtive values
cumPipeSmokingDeathAvoided_3Yminus(9:end,:) =cumPipeSmokingDeathAvoided_nonNegative(9:end,:)-cumPipeSmokingDeathAvoided_nonNegative(9,:); % row 8 is 2028 which is 3rd year of the policy

cumPipeSmokingDeathAvoided_3Yminus_prcnt= prctile(cumPipeSmokingDeathAvoided_3Yminus,[5,50,95],2); %[5,50,95] Percentile

% Exporting to an Excel file:
Year = (2021:2100)';
T = table(Year,PipeSmokingDeath_base, cumPipeSmokingDeathAvoided_3Yminus_prcnt);
writetable(T,filename,'Sheet',"MortalityReducedOtherDeaths",'Range','Y4:AC83','WriteVariableNames',0);


% ===== This was used for  Flavored Cigar mortality, but now it is not used.
% Potential Impact of a Nicotine Product Standard after considering Ban of Flavored Cigars
% % Table: Cumulative Non-Premium Cigar Attributable Deaths Avoided
% prematuredeaths_allCigar= 9246;  % Nonnemaker et al. all cigar premature deaths
% %            % 80% reported non-premium cigars (i.e., little filtered cigars or non-premium cigarillos/large cigars)
% %      % and 20% reported premium cigars as their usual cigar type using a classification methodology described previously (Corey et al., 2014)
% NonPrmiumCigar = 0.8*prematuredeaths_allCigar;  %So NonPrmiumCigar = 7396  Non- premium cigar premature deaths
% NonPrmiumCigarNoFlavor= NonPrmiumCigar - 780; %  since Mortality Reduction from Cigar Flavor Bani is 780 ((Nonnemaker et al., 2014).  )
% % Non- premium cigar premature deaths after Cigar flavor Ban is 6616.8;
% DirectAdult_TADs = 437400; % Direct Adult Smoking-Attributable Deaths, 2005-2009

% NonPrmiumCigarNoFlavorDeath_base = (NonPrmiumCigarNoFlavor/DirectAdult_TADs).*b.padP1; %%sum(c.padP1_vec_f{1} + c.padP1_vec_m{1},1)';
% NonPrmiumCigarNoFlavorDeath_poli = (NonPrmiumCigarNoFlavor/DirectAdult_TADs).*sa_output.padP1{1}; %sum(c.padP1_vec_f{2} + c.padP1_vec_m{2},1)';
% cumNonPrmiumCigarNoFlavorDeathAvoided =cumsum(bsxfun(@minus,NonPrmiumCigarNoFlavorDeath_base,NonPrmiumCigarNoFlavorDeath_poli ));
% cumNonPrmiumCigarNoFlavorDeathAvoided_3Yminus =zeros(size(cumNonPrmiumCigarNoFlavorDeathAvoided));
% cumNonPrmiumCigarNoFlavorDeathAvoided_nonNegative =max(cumNonPrmiumCigarNoFlavorDeathAvoided, 0);  % removing negtive values
% cumNonPrmiumCigarNoFlavorDeathAvoided_3Yminus(8:end,:) =cumNonPrmiumCigarNoFlavorDeathAvoided_nonNegative(8:end,:)-cumNonPrmiumCigarNoFlavorDeathAvoided_nonNegative(8,:); % row 8 is 2028 which is 3rd year of the policy

% cumNonPrmiumCigarNoFlavorDeathAvoided_3Yminus_prcnt= prctile(cumNonPrmiumCigarNoFlavorDeathAvoided_3Yminus,[5,50,95],2); %[5,50,95] Percentile
% % Exporting to an Excel file:
% Year = (2021:2100)';
% T = table(Year, NonPrmiumCigarNoFlavorDeath_base, cumNonPrmiumCigarNoFlavorDeathAvoided_3Yminus_prcnt);
% writetable(T,filename,'Sheet',"MortalityReducedOtherDeaths",'Range','AE4:AI83','WriteVariableNames',0);

%% ------ Code to export additional tables from model outputs -----------


Year = (2021:2100)';

% ------ Sheet: Initiation and Cessation Rates -------
% Add Year  on Column A of the new excel sheet
T = table(Year);
writetable(T,filename,'Sheet',"Initiation and Cessation Rates",'Range','A4:A83','WriteVariableNames',0);

% Table for adltcessP1
writetable(sa_tables.adltcessP1,filename,'Sheet',"Initiation and Cessation Rates",'Range','B4:E83','WriteVariableNames',0);
% Table for YouthinitP1
writetable(sa_tables.YouthinitP1,filename,'Sheet',"Initiation and Cessation Rates",'Range','F4:I83','WriteVariableNames',0);
% Table for adltcessP2
writetable(sa_tables.adltcessP2,filename,'Sheet',"Initiation and Cessation Rates",'Range','J4:M83','WriteVariableNames',0);
% Table for YouthinitP2
writetable(sa_tables.YouthinitP2,filename,'Sheet',"Initiation and Cessation Rates",'Range','N4:Q83','WriteVariableNames',0);

% -----Sheet: Prevalence_CurrentFormerNever --------
T = table(Year);
writetable(T,filename,'Sheet',"Prevalence_CurrentFormerNever",'Range','A4:A83','WriteVariableNames',0);

% Table for adultPrev_c1
writetable(sa_tables.adultPrev_c1,filename,'Sheet',"Prevalence_CurrentFormerNever",'Range','B4:E83','WriteVariableNames',0);
% Table for adultPrev_f1
writetable(sa_tables.adultPrev_f1,filename,'Sheet',"Prevalence_CurrentFormerNever",'Range','F4:I83','WriteVariableNames',0);
% Table for adultPrev_n1
writetable(sa_tables.adultPrev_n1,filename,'Sheet',"Prevalence_CurrentFormerNever",'Range','J4:M83','WriteVariableNames',0);
% Table for adultPrev_c2
writetable(sa_tables.adultPrev_c2,filename,'Sheet',"Prevalence_CurrentFormerNever",'Range','N4:Q83','WriteVariableNames',0);
% Table for adultPrev_f2
writetable(sa_tables.adultPrev_f2,filename,'Sheet',"Prevalence_CurrentFormerNever",'Range','R4:U83','WriteVariableNames',0);
% Table for adultPrev_n2
writetable(sa_tables.adultPrev_n2,filename,'Sheet',"Prevalence_CurrentFormerNever",'Range','V4:Y83','WriteVariableNames',0);
% Table for adultPrev_cc
writetable(sa_tables.adultPrev_cc,filename,'Sheet',"Prevalence_CurrentFormerNever",'Range','Z4:AC83','WriteVariableNames',0);
% Table for adultPrev_ctot
writetable(sa_tables.adultPrev_ctot,filename,'Sheet',"Prevalence_CurrentFormerNever",'Range','AD4:AG83','WriteVariableNames',0);

% -----Sheet: PopSize_CurrentFormerNever --------
T = table(Year);
writetable(T,filename,'Sheet',"PopSize_CurrentFormerNever",'Range','A4:A83','WriteVariableNames',0);

% Table for adultPop_c1
writetable(sa_tables.adultPop_c1,filename,'Sheet',"PopSize_CurrentFormerNever",'Range','B4:E83','WriteVariableNames',0);
% Table for adultPop_f1
writetable(sa_tables.adultPop_f1,filename,'Sheet',"PopSize_CurrentFormerNever",'Range','F4:I83','WriteVariableNames',0);
% Table for adultPop_n1
writetable(sa_tables.adultPop_n1,filename,'Sheet',"PopSize_CurrentFormerNever",'Range','J4:M83','WriteVariableNames',0);
% Table for adultPop_c2
writetable(sa_tables.adultPop_c2,filename,'Sheet',"PopSize_CurrentFormerNever",'Range','N4:Q83','WriteVariableNames',0);
% Table for adultPop_f2
writetable(sa_tables.adultPop_f2,filename,'Sheet',"PopSize_CurrentFormerNever",'Range','R4:U83','WriteVariableNames',0);
% Table for adultPop_n2
writetable(sa_tables.adultPop_n2,filename,'Sheet',"PopSize_CurrentFormerNever",'Range','V4:Y83','WriteVariableNames',0);
% Table for adultPop_cc
writetable(sa_tables.adultPop_cc,filename,'Sheet',"PopSize_CurrentFormerNever",'Range','Z4:AC83','WriteVariableNames',0);

% -----Sheet: Mortality --------
% Add Year  on Column A of the new excel sheet
T = table(Year);
writetable(T,filename,'Sheet',"Mortality",'Range','A4:A83','WriteVariableNames',0);

% Tobacco Attributable Deaths (TADs)
writetable(sa_tables.ad,filename,'Sheet',"Mortality",'Range','B4:E83','WriteVariableNames',0);
% Annual Difference in TADs (scenario - baseline)
writetable(sa_tables.Difad,filename,'Sheet',"Mortality",'Range','F4:H83','WriteVariableNames',0);
% Cumulative sum of TADs annual difference
writetable(sa_tables.cumsumDifad,filename,'Sheet',"Mortality",'Range','I4:K83','WriteVariableNames',0);
% Life years gained
writetable(sa_tables.lifeyears,filename,'Sheet',"Mortality",'Range','L4:O83','WriteVariableNames',0);
% Annual Difference life years gained  (scenario - baseline)
writetable(sa_tables.DifLifeyears,filename,'Sheet',"Mortality",'Range','P4:R83','WriteVariableNames',0);
% Cumulative sum of life years gained difference
writetable(sa_tables.cumsumDifLifeyears,filename,'Sheet',"Mortality",'Range','S4:U83','WriteVariableNames',0);
% QALYs lost
writetable(sa_tables.QALYLost,filename,'Sheet',"Mortality",'Range','V4:Y83','WriteVariableNames',0);
% Annual Difference QALYs lost (scenario - baseline)
writetable(sa_tables.DifQALYLost,filename,'Sheet',"Mortality",'Range','Z4:AB83','WriteVariableNames',0);
% Cumulative  difference QALYs lost
writetable(sa_tables.cumsumDifQALYLost,filename,'Sheet',"Mortality",'Range','AC4:AE83','WriteVariableNames',0);

% -------------- Sheet: Baseline-Smokers by Age --------------
% Add Year  on Column A of the new excel sheet
T = table(Year);

writetable(T,filename,'Sheet',"Baseline-Smokers by Age",'Range','A4:A83','WriteVariableNames',0);
writetable(sa_tables.smokers_0to100years,filename,'Sheet',"Baseline-Smokers by Age",'Range','B4:CX83','WriteVariableNames',0);

% -------------- Sheet: Low-Smokers by Age ---------------
writetable(T,filename,'Sheet',"Policy 5th Perc-Smokers by Age",'Range','A4:A83','WriteVariableNames',0);
writetable(sa_tables.smokers_0to100years_5thPer,filename,'Sheet',"Policy 5th Perc-Smokers by Age",'Range','B4:CX83','WriteVariableNames',0);
% -------------- Sheet: Med-Smokers by Age ---------------
writetable(T,filename,'Sheet',"Policy Median-Smokers by Age",'Range','A4:A83','WriteVariableNames',0);
writetable(sa_tables.smokers_0to100years_median,filename,'Sheet',"Policy Median-Smokers by Age",'Range','B4:CX83','WriteVariableNames',0);
% -------------- Sheet: High-Smokers by Age ---------------
writetable(T,filename,'Sheet',"Policy 95th Per-Smokers by Age",'Range','A4:A83','WriteVariableNames',0);
writetable(sa_tables.smokers_0to100years_95thPer,filename,'Sheet',"Policy 95th Per-Smokers by Age",'Range','B4:CX83','WriteVariableNames',0);

% ---------------Sheet: Ext NonC Initiates and Quitters----------------
T = table(Year);
writetable(T,filename,'Sheet',"Ext NonC Initiates and Quitters",'Range','A4:A83','WriteVariableNames',0);

% New NonCombustible Initiates (includes switchers)
writetable(sa_tables.Count_initP2,filename,'Sheet',"Ext NonC Initiates and Quitters",'Range','B4:E83','WriteVariableNames',0);
% Annual Difference in New Initiates (scenario - baseline)
writetable(sa_tables.DifCount_initP2,filename,'Sheet',"Ext NonC Initiates and Quitters",'Range','F4:H83','WriteVariableNames',0);
% Cumulative Difference in New Initiates (scenario - baseline)
writetable(sa_tables.cumsumDifCount_initP2,filename,'Sheet',"Ext NonC Initiates and Quitters",'Range','I4:K83','WriteVariableNames',0);
% NonCombustible Quitters
writetable(sa_tables.Count_cessp2,filename,'Sheet',"Ext NonC Initiates and Quitters",'Range','L4:O83','WriteVariableNames',0);
% Annual Difference in Quitters (scenario - baseline)
writetable(sa_tables.DifCount_cessp2,filename,'Sheet',"Ext NonC Initiates and Quitters",'Range','P4:R83','WriteVariableNames',0)
% Cumulative Difference in Quitters (scenario - baseline)
writetable(sa_tables.cumsumDifCount_cessp2,filename,'Sheet',"Ext NonC Initiates and Quitters",'Range','S4:U83','WriteVariableNames',0);

% ---------------Sheet: Ext Mortality----------------
% Add Year  on Column A of the new excel sheet
T = table(Year);
writetable(T,filename,'Sheet',"Ext Mortality",'Range','A4:A83','WriteVariableNames',0);
% Annual  Deaths (Ages 35-84)
writetable(sa_tables.age_deaths35to84,filename,'Sheet',"Ext Mortality",'Range','B4:E83','WriteVariableNames',0);
% Annual Difference in Deaths (Ages 35-84) (scenario - baseline)
writetable(sa_tables.DifAge_deaths35to84,filename,'Sheet',"Ext Mortality",'Range','F4:H83','WriteVariableNames',0);
% Cumulative Difference in Deaths (35-84) (scenario - baseline)
writetable(sa_tables.cumsumDifAge_deaths35to84,filename,'Sheet',"Ext Mortality",'Range','I4:K83','WriteVariableNames',0);
% QALYs lived by the adults in the population for the ith year
writetable(sa_tables.adult_qaly,filename,'Sheet',"Ext Mortality",'Range','L4:O83','WriteVariableNames',0);
% Annual Difference  QALYS lived (scenario - baseline)
writetable(sa_tables.DifAdult_qaly,filename,'Sheet',"Ext Mortality",'Range','P4:R83','WriteVariableNames',0);
% Cumulative Difference  QALYS lived (scenario - baseline)
writetable(sa_tables.cumsumDifAdult_qaly,filename,'Sheet',"Ext Mortality",'Range','S4:U83','WriteVariableNames',0);

%%===============================================================================



disp('Exporting done!');

